/*
 *            File: ServerLotto.java
 *          Autore: Roberto FULIGNI
 * Ultima modifica: 20/11/2023
 *
 *     Descrizione: Problema n. 5 pag. 135 del libro di testo
 *                  (sistema di estrazioni del lotto - modulo server)
 */

import edu.fauser.netlab.UniqueRandom;

import java.io.*;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.ArrayList;
import java.util.Collections;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

public class ServerLotto {
    final static int PORTA_SERVER = 5060;
    final static int MAX_GIOCATORI = 2;         // Numero di client da attendere prima di procedere all'estrazione
    final static int MAX_NUMERI_LOTTO = 10;     // Valore massimo dei numeri del lotto (in generale 90, può
                                                // essere ridotto durante i test)
    static ServerSocket globalServerSck;        // variabile condivisa, conterrà il riferimento al server socket
    static ExecutorService pool = Executors.newCachedThreadPool();

    static ArrayList<Integer> numeriVincenti = new ArrayList<Integer>();
    static CountDownLatch giocatoriAttesi = new CountDownLatch(MAX_GIOCATORI);
    static CountDownLatch estrazioniAttese = new CountDownLatch(1);

    public static void main(String[] args) {
        try (ServerSocket server = new ServerSocket(PORTA_SERVER)) {
            globalServerSck = server;               // La variabile globalServerSck è utilizzata dal thread di estrazione
            pool.execute(() -> eseguiEstrazione());
            System.out.format("Ricevitoria lotto in ascolto su: %s%n", server.getLocalSocketAddress());

            // Il server socket sarà chiuso dal thread di estrazione
            while(!server.isClosed()) {
                Socket tempSck;
                try {
                    tempSck = server.accept();
                    pool.execute(() -> gestisciClient(tempSck));
                }
                catch (IOException e) {
                    if (!server.isClosed()) {
                        System.err.println(String.format("Errore nella gestione di nuove connessioni:%s", e.getMessage()));
                    }
                }
            }

        } catch (IOException e) {
            System.err.println(String.format("Errore del server: %s", e.getMessage()));
        }

        pool.shutdown();
        try {
            pool.awaitTermination(Long.MAX_VALUE, TimeUnit.NANOSECONDS);
            System.out.println("Programma ricevitoria terminato");
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    private static void eseguiEstrazione() {
        try {
            giocatoriAttesi.await();

            globalServerSck.close();    // Sblocca il main thread, in attesa sul metodo accept()
            System.out.format("%nRicevitoria chiusa, non si accettano più giocatori...%n");

            System.out.println("Estrazione dei cinque numeri vincenti");
            var ur = new UniqueRandom(1, MAX_NUMERI_LOTTO+1, 2023);
            numeriVincenti.clear();
            for (int i = 0; i < 5; i++) {
                numeriVincenti.add(ur.nextInt());
            }
            Collections.sort(numeriVincenti);
            System.out.print("I numeri estratti sono:");
            for(var v : numeriVincenti)
                System.out.format(" %d", v);
            System.out.println();

        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        estrazioniAttese.countDown();
    }

    private static void gestisciClient(Socket sck) {
        try (var br = new BufferedReader(new InputStreamReader(sck.getInputStream(), "UTF-8"));
             var pw = new PrintWriter(new OutputStreamWriter(sck.getOutputStream(), "UTF-8"), true)) // true -> Autoflush abilitato
        {
            // Fase 1 : si riceve dal client una linea di testo contenente tre numeri interi separati da uno spazio
            var listaNumeri = new ArrayList<Integer>();
            var str = br.readLine();
            if (str == null) {
                sck.close();
                return;
            }
            System.out.format("Ricevuto da %s i seguenti numeri: %s%n",
                    sck.getRemoteSocketAddress(),
                    str);

            var parti = str.split(" ");     // Divide la stringa ricevuta in più parti
            if (parti.length != 3) {
                var errore = "È richiesto l'invio di tre numeri";
                System.err.println(errore);
                pw.println(errore);
                sck.close();
                return;
            }
            for(var p : parti) {
                try {
                    var i = Integer.parseInt(p);
                    listaNumeri.add(i);
                } catch (NumberFormatException e) {
                    var errore = String.format("Numero non valido: %s", p);
                    System.err.println(errore);
                    pw.println(errore);
                    sck.close();
                    return;
                }
            }
            // Il client  ha completato l'invio dei numeri, si riduce il numero di giocatori attesi
            giocatoriAttesi.countDown();
            // Si attende l'estrazione dei numeri vincenti
            estrazioniAttese.await();

            // Fase n. 2: Comunicazione dell'esito dell'estrazione
            var sb = new StringBuilder();
            for(var v: numeriVincenti)
                sb.append(String.format("%d ", v));

            String[] esiti = new String[] {"NESSUN NUMERO ESTRATTO", "UN NUMERO ESTRATTO", "AMBO", "TERNO"};
            var estratti = contaNumeriEstratti(listaNumeri, numeriVincenti);
            sb.append(esiti[estratti]);
            pw.println(sb);
            sck.close();
        } catch (UnsupportedEncodingException e) {
            System.err.println(String.format("Codifica di caratteri non supportata: %s", e.getMessage()));
        } catch (IOException e) {
            System.err.println(String.format("Errore di I/O: %s", e.getMessage()));
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    private static int contaNumeriEstratti(ArrayList<Integer> listaNumeri, ArrayList<Integer> numeriVincenti) {
        int conta = 0;
        for(var n: listaNumeri) {
            if (numeriVincenti.indexOf(n) >= 0)
                conta++;
        }
        return conta;
    }
}