﻿/*
 *         Progetto: Libreria per lo studio dei grafi
 * 
 *           Autore: Roberto Fuligni
 *  Ultima modifica: 06/03/2022
 *  
 *      Descrizione: Gestione dei nodi e dei vertici di un grafo. 
 */

using System.Collections.Generic;
using System.Text;

namespace LibreriaGrafi
{
    public class Grafo
    {
        public const int Infinito = int.MaxValue;
        public const int NessunNodo = int.MinValue;
        public enum Tipo { Orientato, NonOrientato };

        private int numNodi;
        private Tipo tipo;  
        private int[,] pesi;

        // Accesso ai pesi degli archi attraverso gli indici dei nodi sorgente e destinazione
        // Esempio: g[1][3] = 20        Memorizza un arco (orientato oppure non orientato, in base al tipo di grafo)
        //                              di peso 20 dal nodo sorgente 1 al nodo destinazione 3
        public int this[int sorg, int dest]
        {
            get { return pesi[sorg, dest]; }
            set
            {
                pesi[sorg, dest] = value;
                // Un arco non orientato equivale a una coppia di archi orientati
                if (tipo == Grafo.Tipo.NonOrientato)
                    pesi[dest, sorg] = value;
            }
        }

        public Grafo(int numNodi, Tipo tipo = Tipo.Orientato)
        {
            this.numNodi = numNodi;
            this.tipo = tipo;
            pesi = new int[numNodi, numNodi];
            // Inizializzazione dei pesi del grafo
            // All'inizio il grafo non contiene archi -> tutti i pesi sono infiniti
            for (int i = 0; i < numNodi; i++)
                for (int j = 0; j < numNodi; j++)
                    pesi[i, j] = Grafo.Infinito;
        }

        // Calcolo del percorso a costo minimo, a partire da un dato nodo sorgente,
        // mediante l'algorimo di Dijkstra.
        // Il metodo restituisce un nuovo grafo rappresentante il Minimum Spanning Tree
        // elaborato a partire dal nodo indicato.
        public Grafo Dijkstra(int partenza)
        {
            // Definizione dello stato dei nodi mediante vettori paralleli

            int[] costoTot = new int[numNodi];     // Vettore dei costi totali
            int[] pred = new int[numNodi];        // Vettore dei nodi predecessori 

            // Criterio di confronto tra due nodi (utilizzato per stabilire la priorità di elaborazione nella coda)
            // definito mediante statement Lambda.
            //
            // Dati due nodi A e B, la funzione di confronto restituisce un numero negativo se A è elaborato prima di B
            // (A ha priorità maggiore); un numero positivo se A è elaborato dopo B.
            // Il risultato del confronto è calcolato per differenza tra i costi totali di A e B (precedenza ai costi minori)
            // Nel caso di costi uguali, si sceglie di stabilire la priorità in base all'id del nodo (precedenza agli id minori).

            IComparer<int> confrontaNodi = Comparer<int>.Create(
                (a, b) =>
                {
                    int pa = costoTot[a];
                    int pb = costoTot[b];
                    return  (pa != pb ? pa - pb : a - b);
                });

            // Implementazione di una coda con priorità mediante SortedSet.
            // La coda contiene gli id dei nodi in attesa di essere elaborati.
            // Gli elementi inseriti nella coda sono ordinati in base alla priorità:
            // il primo elemento della coda (elemento minimo) corrisponde al
            // nodo con priorità maggiore.

            SortedSet<int> coda = new SortedSet<int>(confrontaNodi);

            // Fase 1: Inizializzazione dello stato e inserimento dei nodi in coda
            for (int i = 0; i < numNodi; i++)
            {
                costoTot[i] = (i == partenza ? 0 : Grafo.Infinito);
                pred[i] = Grafo.NessunNodo;
                coda.Add(i);
            }

            while (coda.Count > 0)
            {
                // Estrazione dalla coda del nodo a costo minimo
                int corrente = coda.Min;
                coda.Remove(corrente);
                
                // Se il nodo estratto ha peso infinito, la coda contiene solo nodi irraggiungibili:
                // il ciclo può essere interrotto.
                if (costoTot[corrente] == Grafo.Infinito)
                    break;

                for (int vicino = 0; vicino < numNodi; vicino++)
                {
                    // Analisi dei nodi vicini (collegati al nodo corrente con un arco)
                    int peso = this[corrente, vicino];
                    if (peso != Grafo.Infinito && costoTot[corrente] + peso < costoTot[vicino])
                    {
                        // Per aggiornare il costo totale, si rimuove il nodo "vicino" dal set,
                        // si ricalcola il suo costo e si reinserisce il nodo aggiornato nel set.
                        // Questa operazioni sono equivalenti all'aggiornamento della priorità
                        // del nodo nella coda.

                        coda.Remove(vicino);
                        costoTot[vicino] = costoTot[corrente] + peso;
                        pred[vicino] = corrente;    // Aggiornamento del nodo predecessore
                        coda.Add(vicino);
                    }
                }
            }

            // Costruzione del grafo MST a partire dagli stati finali dei nodi
            Grafo mst = new Grafo(numNodi, Grafo.Tipo.Orientato);
            for (int i = 0; i < numNodi; i++)
            {
                int h = pred[i];
                // Se il nodo i ha un predecessore, si inserisce nel grafo l'arco pesato h -> i 
                if (h != Grafo.NessunNodo)
                    mst[h, i] = pesi[h, i];
            }
            return mst;
        }

        public override string ToString()
        {
            // Rappresentazione testuale del grafo.
            // Il formato di rappresentazione è compatibile con
            // il software GraphViz (https://graphviz.org/).

            // Strumento online per visualizzare un grafo in formato GraphViz:
            // https://dreampuf.github.io/GraphvizOnline/


            var sb = new StringBuilder();
            string tg = tipo == Grafo.Tipo.Orientato ? "digraph" : "graph";
            string tl = tipo == Grafo.Tipo.Orientato ? "->" : "--";
            sb.AppendLine(tg + " {");
            for (int i = 0; i < numNodi; i++)
            {
                int inizio = tipo == Grafo.Tipo.NonOrientato ? i : 0;
                for (int j = inizio; j < numNodi; j++)
                {
                    int peso = this[i, j];
                    if (peso != Grafo.Infinito)
                        sb.AppendLine($"   {i} {tl} {j} [label={peso}];");
                }
            }

            sb.AppendLine("}");
            return sb.ToString();
        }
    }
}
